% ------------------------------------------------------------------------
% dirmspMHblkgam
% AUTHOR : DAVID KESSLER, dkessler@live.unc.edu
%
% Draw from the GAMMAs instead of the DIRICHLET.
% Draw from their LOGS so that we have something on the real line.
%
% This uses a Metropolis algorithm - that is, a symmetric proposal density
% designed to wiggle the logs of the GAMMAs that make up the DIRICHLET.
%
% Change-of-variables, etc.
%
% Random "bite" size of things that are updated.
% Fixed window (variance) for the Metropolis draw.
%
% marg : contains window information
%        dimension of the problem
%        information content for the marginal prior
%
% mobs : the STACKED observations, corresponding to counts for each element
%        of the "f" vector
% 
% margdist : the source of the prior on the marginals, will be adjusted
%            by tuning parameters to give a prior of the desired
%            informativeness
% ------------------------------------------------------------------------
function mres = dirmspMHblkgam(marg, mobs, margdist)

mres.mrevstr = '$Rev: 99AB $';
mres.mrevstr = mres.mrevstr(7:(end-2));

% Converts joint to marginals via
% marginals = reshape(marger * f, max(marg.DVEC), numel(marg.DVEC))
marger = getmarger(marg.DVEC);

% Indicates which elements of the marginal are for real
keeper = getkeeper(marg.DVEC);

% How many elements
Sp = prod(marg.DVEC);

% Storage for the posterior mean
effsum = zeros(Sp,1);
effcount = 0;

% Target prior, tuned by supplied params
p1 = margdist ./ repmat(sum(margdist,1), size(margdist,1), 1) * marg.P3MARGALPHA;

% Induced marginal prior from (possibly uniform, definitely symmetric) Dirichlet
p0 = reshape(marger * (ones(Sp,1)*marg.P3BASEALPHA), max(marg.DVEC), numel(marg.DVEC));

% Start with a draw from the base prior
gammas = gamrnd(mobs + marg.P3BASEALPHA, 1);

% PROBLEMATIC for very small marg.P3BASEALPHA values, so set a bottom
gammas(gammas<=1e-10) = 1e-10;

dubs = log(gammas);
eff = gammas / sum(gammas);

mres.naccept = 0;
mres.ntrial = 0;

effsamp = zeros(Sp, marg.MAXITER);

% Sample index
tidx = 1;

% Iteration index - may keep on growing...
iteridx = 1;

% Reporting and storage interval
repint = 500;
daccept = 0;
dtrial = 0;

% Gather a specific number of samples
while tidx <= marg.MAXITER
    % Draw a series of updates to the DUB variables via some mechanism
    % exp(DUB_h) \sim Gamma(1,1)

    % The "bite" we update - random SIZE, not adjusted
    bite = 2 ^ randsample(6,1);
    selidx = randsample(Sp, bite);

    % The draw: make sure to use randn() and not rand(), knucklehead
    % Scale variance DOWN with larger bitesizes
    dubdraw = randn(bite,1) * sqrt(marg.MHVAR/bite) + dubs(selidx);

    % The new state
    dubstar = dubs;
    dubstar(selidx) = dubdraw;
    gamstar = gammas;
    gamstar(selidx) = exp(dubdraw);
    
    % NO, don't fix it here
    % gamstar(gamstar<1e-10) = 1e-10;
    effstar = gamstar / sum(gamstar);

    % marginals
    theta = reshape(marger * eff, size(p1));
    thetastar = reshape(marger * effstar, size(p1));

    % Compute logMH
    
    % Marginal prior piece
    if marg.P3ASSUMEDIFFUSE == 1
        logMHa = sum(p1(keeper(:)) .* (log(thetastar(keeper(:))) - log(theta(keeper(:)))));
    else
        logMHa = sum((p1(keeper(:))-p0(keeper(:))) .* (log(thetastar(keeper(:))) - log(theta(keeper(:)))));
    end

    % Posterior piece - compute from the gammas/gamstar
    
    % What I THINK is correct
    TEE = sum(gammas);
    TEESTAR = sum(gamstar);
    logMHb = sum((mobs + marg.P3BASEALPHA) .* (dubstar - dubs)) + (TEE - TEESTAR) + sum(mobs) * (log(TEE) - log(TEESTAR));
    
    % Matching Peter (Hoff)'s implementation - doesn't make much of a difference
    % but I'd prefer to follow what I believe to be true
    % Also, the log-pdf stuff hits a wall sometimes
    % logMHbX = sum(log(gampdf(gamstar, 1 + mobs, 1)) - log(gampdf(gammas, 1+mobs, 1))) + sum(dubstar) - sum(dubs);
    % logMHbX = sum(gammas) - sum(gamstar) + sum((mobs + marg.P3BASEALPHA - 1) .* (log(gamstar) - log(gammas))) + sum(dubstar) - sum(dubs);
    
    % Proposal distribution is SYMMETRIC: P(dubstar; dub) = P(dub; dubstar)

    % total
    logMH = logMHa + logMHb;

    % Accept or reject
    mres.ntrial = mres.ntrial + 1;
    dtrial = dtrial + 1;

    munif = rand(1);
    if log(munif) < logMH
        mres.naccept = mres.naccept + 1;
        daccept = daccept + 1;
        gammas = gamstar;
        eff = effstar;
        dubs = dubstar;
    end

    % Tricky since we don't know how many there will be, but whatever
    if mod(iteridx, repint) == 0
        LogIt(sprintf('MHAR: [%d/%d=%3.3e] [%d/%d=%3.3e] iteridx=%d\n', ...
            mres.naccept, mres.ntrial, ...
            mres.naccept / mres.ntrial, ...,
            daccept, dtrial, daccept/dtrial, iteridx), 3, marg.LOGTHRESH);
        daccept = 0;
        dtrial = 0;

        % Save the current probability vector
        % "EFF" is the way you pronounce the letter "F".
        if iteridx / repint > marg.BURNIN
            effsamp(:,tidx) = eff;
            tidx = tidx + 1;

            % progress?
            if mod(tidx, floor(marg.MAXITER/100)) == 0
                LogIt(sprintf('ITER: %d / %d\n', tidx, marg.MAXITER), 3, marg.LOGTHRESH);
            end
        end
        
    end
    
    % ALWAYS add to the total
    effsum = effsum + eff;
    effcount = effcount + 1;
    
    % Keep track of what we've done
    iteridx = iteridx + 1;
end

% Save the results
mres.accrate = mres.naccept / mres.ntrial;
mres.postmean = effsum / effcount;
mres.effsamp = effsamp;

end
