% ------------------------------------------------------------------------
% AUTHOR : DAVID KESSLER, dkessler@live.unc.edu
% Convert an index into a TENSOR into the set of indices into the
% decomposed, independent vectors.  That is, the tensor is:
% pi = psi_1 \kron psi_2 \kron \ldots \kron \psi_p
% pi is of size S = \prod_{j=1}^p d_j
% Convert an index in the range 1 \ldots S into a p-element vector, where
% the jth element represents the index into the j^th \psi vector.
% ------------------------------------------------------------------------
function ovec = idx2vec(midxlist, idvec)

dvec = reshape(idvec, 1, numel(idvec));

% Set up the result
mvec = zeros(length(midxlist),length(dvec));
maxidx = prod(dvec);

% Convert to zero-based index to make the computations easier
idxlist = midxlist(:) - 1;

% Knock out bad entries - trouble for caller!
idxlist(idxlist >= maxidx) = 0;
idxlist(idxlist <  0)      = 0;

% Create the power list
baselist = dvec(2:end);
powlist = [1 cumprod(baselist(end:-1:1))];
powlist = powlist(end:-1:1);

mvec(:,1) = floor(idxlist / powlist(1));
rvec = idxlist - mvec(:,1) * powlist(1);

for jidx = 2 : length(powlist)-1
    mvec(:,jidx) = floor(rvec / powlist(jidx));
    rvec = rvec - mvec(:,jidx) * powlist(jidx);
end

if length(powlist) > 1
    mvec(:,length(dvec)) = rvec;
end

% Go back to one-based index
ovec = mvec + 1;

badidx = midxlist>maxidx | midxlist < 1;
if any(badidx)
    ovec(badidx, :) = ones(1,size(ovec,2))*(-1);
end

end
